<?php

namespace CleantalkSP\SpbctWP\Scanner\FileOfPluginCheckerModule;

/**
 * Represents the verdict/result of plugin source analysis.
 *
 * This class stores information about a scanned plugin including its metadata,
 * scan success status, and any messages generated during the scanning process.
 */
class FileOfPluginCheckerVerdict
{
    /**
     * Version of the plugin.
     *
     * @var string
     */
    public $version = 'unknown';

    /**
     * Slug identifier of the plugin.
     *
     * @var string
     */
    public $slug = 'unknown';

    /**
     * Indicates if there was an initialization error.
     *
     * @var bool
     */
    public $init_error = false;

    /**
     * Indicates if the scan was successful.
     *
     * @var bool
     */
    public $success = false;

    /**
     * The main plugin root file.
     *
     * @var string
     */
    private $root_file = 'unknown';

    /**
     * Path to the plugins directory.
     *
     * @var string
     */
    private $plugins_dir = 'unknown';

    /**
     * Collection of messages generated during scanning.
     *
     * @var array
     */
    private $messages = [];

    /**
     * Sets the plugin version.
     *
     * @param mixed $version The version string to set.
     */
    public function setVersion($version)
    {
        is_scalar($version) && $this->version = (string)$version;
    }

    /**
     * Sets the plugins directory path.
     *
     * @param mixed $plugins_dir The plugins directory path.
     */
    public function setPluginsDir($plugins_dir)
    {
        is_scalar($plugins_dir) && $this->plugins_dir = (string)$plugins_dir;
    }

    /**
     * Sets the plugin slug.
     *
     * @param mixed $slug The plugin slug identifier.
     */
    public function setSlug($slug)
    {
        is_scalar($slug) && $this->slug = (string)$slug;
    }

    /**
     * Sets the plugin root file name.
     *
     * @param mixed $root_file_name The main plugin file name.
     */
    public function setRootFile($root_file_name)
    {
        is_scalar($root_file_name) && $this->root_file = (string)$root_file_name;
    }

    /**
     * Marks the scan as successful and returns the instance.
     *
     * @return $this
     */
    public function success()
    {
        $this->success = true;
        return $this;
    }

    /**
     * Adds a message to the verdict with optional file path context.
     *
     * @param string $string The message text.
     * @param string $file_path The file path associated with the message.
     */
    public function addMessage($string, $file_path = 'unknown')
    {
        $msg_array = array(
            'normalized_path' => $file_path,
            'message' => $string,
        );
        $this->messages[] = $msg_array;
    }

    /**
     * Converts the scan information to JSON format.
     *
     * @return string JSON representation of scan information or error string.
     */
    public function getScanInfoJSON()
    {
        $array = [
            'init_error'  => $this->init_error,
            'success'     => $this->success,
            'slug'        => $this->slug,
            'version'     => $this->version,
            'root_file'   => $this->root_file,
            'plugins_dir' => $this->plugins_dir,
            'messages'   => $this->messages,
        ];

        $json = json_encode($array);

        return false !== $json ? $json : 'JSON_ENCODE_ERROR';
    }
}
